function cdx_new = convert_daily2monthly(CDX)
% --------------------------------------------------------------------------------------------------
% Convert a credit index structure from daily data to monthly data, using the first business day of
% each month.
% --------------------------------------------------------------------------------------------------
% CDX               ... credit index structure (see 'all_steps_in_a_row.m')
% --------------------------------------------------------------------------------------------------
% sample call: convert_daily2monthly(CDX_NA_IG)
% --------------------------------------------------------------------------------------------------

% Determine first business day of each month
business_days = CDX.dates{1};
date_details = datevec(business_days);
used_rows = find(diff(date_details(:,2)) ~= 0)+1;
if (date_details(used_rows(1),2) ~= date_details(1,2))
    used_rows = [1; used_rows];
end
used_dates = business_days(used_rows);
used_dates = datenum(used_dates);
cdx_new = CDX;

% Convert CDS price data to monthly frequency
for i=1:length(cdx_new.portfolio)
    cds = cdx_new.portfolio(i);
    for j=1:length(cds.dates)
        % Determine relevant rows
        [trash, used_rows] = is_member_sorted_c(used_dates, cds.dates{j});
        used_rows = used_rows(find(used_rows>0));
        
        % Select relevant data rows
        cds.dates{j} = cds.dates{j}(used_rows,:);
        cds.T{j} = cds.T{j}(used_rows,:);
        cds.market_price{j} = cds.market_price{j}(used_rows,:);
        if (isfield(cds, 'bid_ask_spread'))
            if (length(cds.bid_ask_spread{j}) > 0)
                cds.bid_ask_spread{j} = cds.bid_ask_spread{j}(used_rows,:);
            end
        end
        if (isfield(cds, 'x0') & (j==1) & (length(cds.x0)>0))
            cds.x0 = cds.x0(used_rows,:);
        end
        if (isfield(cds, 'debt') & (length(cds.debt)>0) & (j==1))
            cds.debt = cds.debt(used_rows);
        end
        if (isfield(cds, 'mcap') & (length(cds.mcap)>0) & (j==1))
            cds.mcap = cds.mcap(used_rows);
        end
        if (isfield(cds, 'share_price') & (length(cds.share_price)>0) & (j==1))
            cds.share_price = cds.share_price(used_rows);
        end
        if (isfield(cds, 'intensity_P') & (length(cds.intensity_P)>0) & (j==1))
            cds.intensity_P = cds.intensity_P(used_rows);
        end
        if (isfield(cds, 'rating') & (length(cds.rating)>0) & (j==1))
            cds.rating = cds.rating(used_rows);
        end
    end
    cdx_new.portfolio(i) = cds;
end

% Convert index and tranche price data to monthly frequency
for i=1:length(CDX.dates)
    % Determine relevant rows
    [trash, used_rows] = is_member_sorted_c(used_dates, CDX.dates{i});
    used_rows = used_rows(find(used_rows>0));  
    
    % Select relevant data rows (for each maturity of index)
    cdx_new.dates{i} = cdx_new.dates{i}(used_rows,:);
    cdx_new.T{i} = cdx_new.T{i}(used_rows,:);    
    if (isfield(CDX, 'series_name'))
        cdx_new.series_name{i} = cdx_new.series_name{i}(used_rows,:);
    end
    if (isfield(CDX, 'series_id'))
        cdx_new.series_id{i} = cdx_new.series_id{i}(used_rows,:);
    end
    cdx_new.index_market_price{i} = cdx_new.index_market_price{i}(used_rows,:);
    if (isfield(CDX, 'tranche_market_price_bid'))
        cdx_new.tranche_market_price_bid{i} = cdx_new.tranche_market_price_bid{i}(used_rows,:);
    end
    if (isfield(CDX, 'tranche_market_price_ask'))
        cdx_new.tranche_market_price_ask{i} = cdx_new.tranche_market_price_ask{i}(used_rows,:);
    end
    cdx_new.tranche_market_price_mid{i} = cdx_new.tranche_market_price_mid{i}(used_rows,:);
    if (isfield(CDX, 'tranche_market_upfront_bid'))
        cdx_new.tranche_market_upfront_bid{i} = cdx_new.tranche_market_upfront_bid{i}(used_rows,:);
    end
    if (isfield(CDX, 'tranche_market_upfront_ask'))
        cdx_new.tranche_market_upfront_ask{i} = cdx_new.tranche_market_upfront_ask{i}(used_rows,:);
    end
    cdx_new.tranche_market_upfront_mid{i} = cdx_new.tranche_market_upfront_mid{i}(used_rows,:);
    
    % Select (general) relevant data rows
    if (i == 1)
        if (isfield(CDX, 'y0'))
            cdx_new.y0 = cdx_new.y0(used_rows,:);
        end
        cdx_new.index_members = cdx_new.index_members(used_rows,:);
    end
    
    % Also convert model-implied prices
    if (isfield(CDX, 'tranche_model_upfront'))
        if (length(CDX.tranche_model_upfront) >= i)
            cdx_new.tranche_model_upfront{i} = cdx_new.tranche_model_upfront{i}(used_rows,:);
        end
    end
    if (isfield(CDX, 'tranche_model_price'))
        if (length(CDX.tranche_model_price) >= i)
            cdx_new.tranche_model_price{i} = cdx_new.tranche_model_price{i}(used_rows,:);
        end
    end
end




